/*
    Copyright (c) 2010 Volker Krause <vkrause@kde.org>

    This library is free software; you can redistribute it and/or modify it
    under the terms of the GNU Library General Public License as published by
    the Free Software Foundation; either version 2 of the License, or (at your
    option) any later version.

    This library is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
    License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to the
    Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include <QtTest/QTest>
#include "../imapparser_p.h"

using namespace Akonadi;

Q_DECLARE_METATYPE( QList<QByteArray> )

class ImapParserBenchmark : public QObject
{
  Q_OBJECT
  private Q_SLOTS:
    void quote_data()
    {
      QTest::addColumn<QByteArray>( "input" );
      QTest::newRow( "empty" ) << QByteArray();
      QTest::newRow( "10-idle" ) << QByteArray( "ababababab" );
      QTest::newRow( "10-quote" ) << QByteArray( "\"abababab\"" );
      QTest::newRow( "50-idle" ) << QByteArray(  "ababababababababababababababababababababababababab" );
      QTest::newRow( "50-quote" ) << QByteArray( "\"abababab\ncabababab\ncabababab\ncabababab\ncabababab\"" );
    }

    void quote()
    {
      QFETCH( QByteArray, input );
      QBENCHMARK {
        ImapParser::quote( input );
      }
    }

    void join_data()
    {
      QTest::addColumn<QList<QByteArray> >( "list" );
      QTest::newRow( "empty" ) << QList<QByteArray>();
      QTest::newRow( "single" ) << ( QList<QByteArray>() << "ababab" );
      QTest::newRow( "two" ) << ( QList<QByteArray>() << "ababab" << "ababab" );
      QTest::newRow( "five" ) << ( QList<QByteArray>() << "ababab" << "ababab" << "ababab" << "ababab" << "ababab" );
      QList<QByteArray> list;
      for ( int i = 0; i < 50; ++i ) {
        list << "ababab";
      }
      QTest::newRow( "a lot" ) << list;
    }

    void join()
    {
      QFETCH( QList<QByteArray>, list );
      QBENCHMARK {
        ImapParser::join( list, " " );
      }
    }

    void parseParenthesizedList_data()
    {
      QTest::addColumn<QByteArray>( "data" );
      QTest::newRow( "empty" ) << QByteArray();
      QTest::newRow( "unnested" ) << QByteArray("(\"Foo Bar\" NIL \"foobar\" \"test.com\")");
      QTest::newRow( "nested" ) << QByteArray("((\"Foo Bar\" NIL \"foobar\" \"test.com\"))");
      QTest::newRow( "nested-long" ) << QByteArray("(UID 86 REV 0 MIMETYPE \"message/rfc822\" COLLECTIONID 13 SIZE 6114 FLAGS (\\SEEN)"
                                                   " ANCESTORS ((13 \"/INBOX\") (12 \"imap://mail@mail.test.com/\") (0 \"\")) PLD:ENVELOPE[1] {396}"
                                                   " (\"Fri, 04 Jun 2010 09:07:54 +0200\" \"Re: [ADMIN] foobar available again!\""
                                                   " ((\"Foo Bar\" NIL \"foobar\" \"test.com\"))"
                                                   " NIL NIL"
                                                   " ((\"Asdf Bla Blub\" NIL \"asdf.bla.blub\" \"123test.org\"))"
                                                   " ((NIL NIL \"muh.kuh\" \"lalala.com\") (\"Konqi KDE\" NIL \"konqi\" \"kde.org\") (NIL NIL \"all\" \"test.com\"))"
                                                   " NIL \"<201006040905.33367.foo.bar@test.com>\" \"<4C08A64A.9020205@123test.org>\""
                                                   " \"<201006040142.56540.muh.kuh@lalala.com> <201006040704.39648.konqi@kde.org> <201006040905.33367.foo.bar@test.com>\""
                                                   "))");
    }

    void parseParenthesizedList()
    {
      QFETCH( QByteArray, data );
      QVarLengthArray<QByteArray, 16> result;
      QBENCHMARK {
        ImapParser::parseParenthesizedList( data, result, 0 );
      }
    }
};

#include "imapparserbenchmark.moc"

QTEST_APPLESS_MAIN( ImapParserBenchmark )
