#! /usr/bin/python3
# -*- coding: utf-8 -*-

# Copyright (C) 2013 David Callé <davidc@framli.eu>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from gi.repository import Unity, UnityExtras
from gi.repository import Gio, GLib
from lxml import etree
import urllib.parse
import urllib.request
import gettext

APP_NAME = 'unity-scope-gallica'
LOCAL_PATH = '/usr/share/locale/'
gettext.bindtextdomain(APP_NAME, LOCAL_PATH)
gettext.textdomain(APP_NAME)
_ = gettext.gettext

GROUP_NAME = 'com.canonical.Unity.Scope.Books.Gallica'
UNIQUE_PATH = '/com/canonical/unity/scope/books/gallica'
SEARCH_URI = 'http://gallica.bnf.fr/'
SEARCH_HINT = _('Search Gallica')
NO_RESULTS_HINT = _('Sorry, there are no Gallica results that match your search.')
PROVIDER_CREDITS = _('Powered by Gallica')
SVG_DIR = '/usr/share/icons/unity-icon-theme/places/svg/'
PROVIDER_ICON = SVG_DIR+'service-gallica.svg'
DEFAULT_RESULT_ICON = SVG_DIR+'service-gallica.svg'
DEFAULT_RESULT_MIMETYPE = 'text/html'
DEFAULT_RESULT_TYPE = Unity.ResultType.DEFAULT

c1 = {'id'      :'top',
      'name'    :_('Books and historical documents'),
      'icon'    :SVG_DIR + 'group-books.svg',
      'renderer':Unity.CategoryRenderer.VERTICAL_TILE}
CATEGORIES = [c1]
FILTERS = []
m1 = {'id'   :'author',
      'type' :'s',
      'field':Unity.SchemaFieldType.OPTIONAL}
m2 = {'id'   :'publisher',
      'type' :'s',
      'field':Unity.SchemaFieldType.OPTIONAL}
m3 = {'id'   :'published_date',
      'type' :'s',
      'field':Unity.SchemaFieldType.OPTIONAL}
EXTRA_METADATA = [m1, m2, m3]

def search(search, filters):
    '''
    Any search method returning results as a list of tuples.
    Available tuple fields:
    uri (string)
    icon (string)
    title (string)
    comment (string)
    dnd_uri (string)
    mimetype (string)
    category (int)
    result_type (Unity ResultType)
    extras metadata fields (variant)
    '''
    results = []
    if not search:
        return results
    search = search.replace(' ', ' AND ')
    search = urllib.parse.quote(search)
    uri = ('%sSRU?operation=searchRetrieve&version=1.2&maximumRecords=25&startRecord=1&query=%s' % (SEARCH_URI, search))
    print (uri)
    try:
        response = urllib.request.urlopen(uri).read()
        data = etree.fromstring(response)
    except Exception as error:
        print (error)
        data = None
        return results
    if data is None:
        return results
    for record in data.iter():
        try:
            if record.tag == "{http://www.loc.gov/zing/srw/}record":
                thumb, title, uri, comment = None, None, None, None
                publisher, published_date = '', ''
                author = []
                for r in record:
                    if r.tag == "{http://www.loc.gov/zing/srw/}extraRecordData":
                        for element in r:
                            if element.tag == "{http://gallica.bnf.fr/namespaces/gallica}thumbnail":
                                if element.text !=  "":
                                    thumb = element.text
                            if element.tag == "{http://gallica.bnf.fr/namespaces/gallica}link":
                                if element.text !=  "":
                                    uri = element.text
                    if r.tag == "{http://www.loc.gov/zing/srw/}recordData":
                        for element in r:
                            if element.tag == "{http://www.openarchives.org/OAI/2.0/oai_dc/}dc":
                                for e in element:
                                    if e.tag == "{http://purl.org/dc/elements/1.1/}title":
                                        if e.text != "":
                                            title = e.text
                                    elif e.tag == "{http://purl.org/dc/elements/1.1/}description":
                                        if e.text != "":
                                            comment = e.text
                                    elif e.tag == "{http://purl.org/dc/elements/1.1/}creator":
                                        if e.text != "":
                                            author.append(e.text)
                                    elif e.tag == "{http://purl.org/dc/elements/1.1/}publisher":
                                        if e.text != "":
                                            publisher = e.text
                                    elif e.tag == "{http://purl.org/dc/elements/1.1/}date":
                                        if e.text != "":
                                            date = e.text
                author = ", ".join(author)
                results.append({'uri':uri,
                                'icon':thumb,
                                'title':title,
                                'comment':comment,
                                'author':author,
                                'publisher':publisher,
                                'published_date':date})
        except Exception as error:  
            print (error)
            continue
    return results



# Classes below this point establish communication
# with Unity, you probably shouldn't modify them.


class MySearch (Unity.ScopeSearchBase):
    def __init__(self, search_context):
        super (MySearch, self).__init__()
        self.set_search_context (search_context)

    def do_run (self):
        '''
        Adds results to the model
        '''
        try:
            result_set = self.search_context.result_set
            for i in search(self.search_context.search_query,
                            self.search_context.filter_state):
                if not 'uri' in i or not i['uri'] or i['uri'] == '':
                    continue
                if not 'icon' in i or not i['icon'] or i['icon'] == '':
                    i['icon'] = DEFAULT_RESULT_ICON
                if not 'mimetype' in i or not i['mimetype'] or i['mimetype'] == '':
                    i['mimetype'] = DEFAULT_RESULT_MIMETYPE
                if not 'result_type' in i or not i['result_type'] or i['result_type'] == '':
                    i['result_type'] = DEFAULT_RESULT_TYPE
                if not 'category' in i or not i['category'] or i['category'] == '':
                    i['category'] = 0
                if not 'title' in i or not i['title']:
                    i['title'] = ''
                if not 'comment' in i or not i['comment']:
                    i['comment'] = ''
                if not 'dnd_uri' in i or not i['dnd_uri'] or i['dnd_uri'] == '':
                    i['dnd_uri'] = i['uri']
                result_set.add_result(**i)
        except Exception as error:
            print (error)

class Preview (Unity.ResultPreviewer):

    def do_run(self):
        icon = Gio.FileIcon.new(Gio.file_new_for_uri(self.result.icon_hint))
        preview = Unity.GenericPreview.new(self.result.title, self.result.comment.strip(), icon)
        preview.props.subtitle = self.result.metadata['author'].get_string()
        if self.result.metadata['publisher'].get_string():
            preview.add_info(Unity.InfoHint.new("publisher", _("Publisher"), None, self.result.metadata['publisher'].get_string()))
        if self.result.metadata['published_date'].get_string():
            preview.add_info(Unity.InfoHint.new("published", _("Published"), None, self.result.metadata['published_date'].get_string()))
#        if self.result.metadata['followers'].get_int32():
#            preview.add_info(Unity.InfoHint.new("followers", _("Followers"), None, str(self.result.metadata['followers'].get_int32())))
        icon = Gio.FileIcon.new (Gio.file_new_for_path(PROVIDER_ICON))
        view_action = Unity.PreviewAction.new("open", _("View"), icon)
        preview.add_action(view_action)
        return preview

class Scope (Unity.AbstractScope):
    def __init__(self):
        Unity.AbstractScope.__init__(self)

    def do_get_search_hint (self):
        return SEARCH_HINT

    def do_get_schema (self):
        '''
        Adds specific metadata fields
        '''
        schema = Unity.Schema.new ()
        if EXTRA_METADATA:
            for m in EXTRA_METADATA:
                schema.add_field(m['id'], m['type'], m['field'])
        #FIXME should be REQUIRED for credits
        schema.add_field('provider_credits', 's', Unity.SchemaFieldType.OPTIONAL)
        return schema

    def do_get_categories (self):
        '''
        Adds categories
        '''
        cs = Unity.CategorySet.new ()
        if CATEGORIES:
            for c in CATEGORIES:
                cat = Unity.Category.new (c['id'], c['name'],
                                          Gio.ThemedIcon.new(c['icon']),
                                          c['renderer'])
                cs.add (cat)
        return cs

    def do_get_filters (self):
        '''
        Adds filters
        '''
        fs = Unity.FilterSet.new ()
#        if FILTERS:
#            
        return fs

    def do_get_group_name (self):
        return GROUP_NAME

    def do_get_unique_name (self):
        return UNIQUE_PATH

    def do_create_search_for_query (self, search_context):
        se = MySearch (search_context)
        return se

    def do_create_previewer(self, result, metadata):
        rp = Preview()
        rp.set_scope_result(result)
        rp.set_search_metadata(metadata)
        return rp

def load_scope():
    return Scope()
