// Copyright 2013 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CHROME_RENDERER_NET_NET_ERROR_HELPER_CORE_H_
#define CHROME_RENDERER_NET_NET_ERROR_HELPER_CORE_H_

#include <string>

#include "base/memory/weak_ptr.h"
#include "chrome/common/net/net_error_info.h"
#include "url/gurl.h"

namespace blink {
struct WebURLError;
}

// Class that contains the logic for how the NetErrorHelper.  This allows for
// testing the logic without a RenderView or WebFrame, which are difficult to
// mock, and for testing races which are impossible to reliably reproduce
// with real RenderViews or WebFrames.
class NetErrorHelperCore {
 public:
  enum FrameType {
    MAIN_FRAME,
    SUB_FRAME,
  };

  enum PageType {
    NON_ERROR_PAGE,
    ERROR_PAGE,
  };

  // The Delegate handles all interaction with the RenderView, WebFrame, and
  // the network, as well as the generation of error pages.
  class Delegate {
   public:
    // Generates an error page's HTML for the given error.
    virtual void GenerateLocalizedErrorPage(const blink::WebURLError& error,
                                            bool is_failed_post,
                                            std::string* html) const = 0;

    // Loads the given HTML in the main frame for use as an error page.
    virtual void LoadErrorPageInMainFrame(const std::string& html,
                                          const GURL& failed_url) = 0;

    // Updates the currently displayed error page with a new error code.  The
    // currently displayed error page must have finished loading, and must have
    // been generated by a call to GenerateLocalizedErrorPage.
    virtual void UpdateErrorPage(const blink::WebURLError& error,
                                 bool is_failed_post) = 0;

    // Fetches an error page and calls into OnErrorPageFetched when done.  Any
    // previous fetch must either be canceled or finished before calling.  Can't
    // be called synchronously after a previous fetch completes.
    virtual void FetchErrorPage(const GURL& url) = 0;

    // Cancels an error page fetch.  Does nothing if no fetch is ongoing.
    virtual void CancelFetchErrorPage() = 0;

   protected:
    virtual ~Delegate() {}
  };

  explicit NetErrorHelperCore(Delegate* delegate);
  ~NetErrorHelperCore();

  // Examines |frame| and |error| to see if this is an error worthy of a DNS
  // probe.  If it is, initializes |error_strings| based on |error|,
  // |is_failed_post|, and |locale| with suitable strings and returns true.
  // If not, returns false, in which case the caller should look up error
  // strings directly using LocalizedError::GetNavigationErrorStrings.
  //
  // Updates the NetErrorHelper with the assumption the page will be loaded
  // immediately.
  void GetErrorHTML(FrameType frame_type,
                    const blink::WebURLError& error,
                    bool is_failed_post,
                    std::string* error_html);

  // These methods handle tracking the actual state of the page.
  void OnStartLoad(FrameType frame_type, PageType page_type);
  void OnCommitLoad(FrameType frame_type);
  void OnFinishLoad(FrameType frame_type);
  void OnStop();

  // Called when an error page have has been retrieved over the network.  |html|
  // must be an empty string on error.
  void OnAlternateErrorPageFetched(const std::string& html);

  // Notifies |this| that network error information from the browser process
  // has been received.
  void OnNetErrorInfo(chrome_common_net::DnsProbeStatus status);

  void set_alt_error_page_url(const GURL& alt_error_page_url) {
    alt_error_page_url_ = alt_error_page_url;
  }

 private:
  struct ErrorPageInfo;

  // Updates the currently displayed error page with a new error based on the
  // most recently received DNS probe result.  The page must have finished
  // loading before this is called.
  void UpdateErrorPage();

  void GenerateLocalErrorPage(
      FrameType frame_type,
      const blink::WebURLError& error,
      bool is_failed_post,
      std::string* error_html);

  blink::WebURLError GetUpdatedError(const blink::WebURLError& error) const;

  Delegate* delegate_;

  // The last DnsProbeStatus received from the browser.
  chrome_common_net::DnsProbeStatus last_probe_status_;

  // Information for the provisional / "pre-provisional" error page.  NULL when
  // there's no page pending, or the pending page is not an error page.
  scoped_ptr<ErrorPageInfo> pending_error_page_info_;

  // Information for the committed error page.  NULL when the committed page is
  // not an error page.
  scoped_ptr<ErrorPageInfo> committed_error_page_info_;

  GURL alt_error_page_url_;
};

#endif  // CHROME_RENDERER_NET_NET_ERROR_HELPER_CORE_H_
