#include <assert.h>
#include <errno.h>
#include <limits.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <algorithm>
#include <string>
#include <vector>

#include "log.h"
#include "metacube2.h"
#include "mutexlock.h"
#include "state.pb.h"
#include "stream.h"
#include "util.h"

using namespace std;

Stream::Stream(const string &url, size_t backlog_size, Encoding encoding)
	: url(url),
	  encoding(encoding),
	  data_fd(make_tempfile("")),
          backlog_size(backlog_size),
	  bytes_received(0),
	  last_suitable_starting_point(-1),
	  mark_pool(NULL),
	  pacing_rate(~0U),
	  queued_data_last_starting_point(-1)
{
	if (data_fd == -1) {
		exit(1);
	}

	pthread_mutex_init(&queued_data_mutex, NULL);
}

Stream::~Stream()
{
	if (data_fd != -1) {
		safe_close(data_fd);
	}
}

Stream::Stream(const StreamProto &serialized, int data_fd)
	: url(serialized.url()),
	  http_header(serialized.http_header()),
	  stream_header(serialized.stream_header()),
	  encoding(Stream::STREAM_ENCODING_RAW),  // Will be changed later.
	  data_fd(data_fd),
	  backlog_size(serialized.backlog_size()),
	  bytes_received(serialized.bytes_received()),
	  mark_pool(NULL),
	  pacing_rate(~0U),
	  queued_data_last_starting_point(-1)
{
	if (data_fd == -1) {
		exit(1);
	}

	assert(serialized.has_last_suitable_starting_point());
	last_suitable_starting_point = serialized.last_suitable_starting_point();

	pthread_mutex_init(&queued_data_mutex, NULL);
}

StreamProto Stream::serialize()
{
	StreamProto serialized;
	serialized.set_http_header(http_header);
	serialized.set_stream_header(stream_header);
	serialized.add_data_fds(data_fd);
	serialized.set_backlog_size(backlog_size);
	serialized.set_bytes_received(bytes_received);
	serialized.set_last_suitable_starting_point(last_suitable_starting_point);
	serialized.set_url(url);
	data_fd = -1;
	return serialized;
}
	
void Stream::set_backlog_size(size_t new_size)
{
	if (backlog_size == new_size) {
		return;
	}

	string existing_data;
	if (!read_tempfile_and_close(data_fd, &existing_data)) {
		exit(1);
	}

	// Unwrap the data so it's no longer circular.
	if (bytes_received <= backlog_size) {
		existing_data.resize(bytes_received);
	} else {
		size_t pos = bytes_received % backlog_size;
		existing_data = existing_data.substr(pos, string::npos) +
			existing_data.substr(0, pos);
	}

	// See if we need to discard data.
	if (new_size < existing_data.size()) {
		size_t to_discard = existing_data.size() - new_size;
		existing_data = existing_data.substr(to_discard, string::npos);
	}

	// Create a new, empty data file.
	data_fd = make_tempfile("");
	if (data_fd == -1) {
		exit(1);
	}
	backlog_size = new_size;

	// Now cheat a bit by rewinding, and adding all the old data back.
	bytes_received -= existing_data.size();
	iovec iov;
	iov.iov_base = const_cast<char *>(existing_data.data());
	iov.iov_len = existing_data.size();

	vector<iovec> iovs;
	iovs.push_back(iov);
	add_data_raw(iovs);
}

void Stream::put_client_to_sleep(Client *client)
{
	sleeping_clients.push_back(client);
}

// Return a new set of iovecs that contains only the first <bytes_wanted> bytes of <data>.
vector<iovec> collect_iovecs(const vector<iovec> &data, size_t bytes_wanted)
{
	vector<iovec> ret;
	size_t max_iovecs = std::min<size_t>(data.size(), IOV_MAX);
	for (size_t i = 0; i < max_iovecs && bytes_wanted > 0; ++i) {
		if (data[i].iov_len <= bytes_wanted) {
			// Consume the entire iovec.
			ret.push_back(data[i]);
			bytes_wanted -= data[i].iov_len;
		} else {
			// Take only parts of this iovec.
			iovec iov;
			iov.iov_base = data[i].iov_base;
			iov.iov_len = bytes_wanted;	
			ret.push_back(iov);
			bytes_wanted = 0;
		}
	}
	return ret;
}

// Return a new set of iovecs that contains all of <data> except the first <bytes_wanted> bytes.
vector<iovec> remove_iovecs(const vector<iovec> &data, size_t bytes_wanted)
{
	vector<iovec> ret;
	size_t i;
	for (i = 0; i < data.size() && bytes_wanted > 0; ++i) {
		if (data[i].iov_len <= bytes_wanted) {
			// Consume the entire iovec.
			bytes_wanted -= data[i].iov_len;
		} else {
			// Take only parts of this iovec.
			iovec iov;
			iov.iov_base = reinterpret_cast<char *>(data[i].iov_base) + bytes_wanted;
			iov.iov_len = data[i].iov_len - bytes_wanted;
			ret.push_back(iov);
			bytes_wanted = 0;
		}
	}

	// Add the rest of the iovecs unchanged.
	ret.insert(ret.end(), data.begin() + i, data.end());
	return ret;
}

void Stream::add_data_raw(const vector<iovec> &orig_data)
{
	vector<iovec> data = orig_data;
	while (!data.empty()) {
		size_t pos = bytes_received % backlog_size;

		// Collect as many iovecs as we can before we hit the point
		// where the circular buffer wraps around.
		vector<iovec> to_write = collect_iovecs(data, backlog_size - pos);
		ssize_t ret;
		do {
			ret = pwritev(data_fd, to_write.data(), to_write.size(), pos);
		} while (ret == -1 && errno == EINTR);

		if (ret == -1) {
			log_perror("pwritev");
			// Dazed and confused, but trying to continue...
			return;
		}
		bytes_received += ret;

		// Remove the data that was actually written from the set of iovecs.
		data = remove_iovecs(data, ret);
	}
}

void Stream::add_data_deferred(const char *data, size_t bytes, StreamStartSuitability suitable_for_stream_start)
{
	MutexLock lock(&queued_data_mutex);
	assert(suitable_for_stream_start == SUITABLE_FOR_STREAM_START ||
	       suitable_for_stream_start == NOT_SUITABLE_FOR_STREAM_START);
	if (suitable_for_stream_start == SUITABLE_FOR_STREAM_START) {
		queued_data_last_starting_point = queued_data.size();
	}

	if (encoding == Stream::STREAM_ENCODING_METACUBE) {
		// Add a Metacube block header before the data.
		metacube2_block_header hdr;
		memcpy(hdr.sync, METACUBE2_SYNC, sizeof(hdr.sync));
		hdr.size = htonl(bytes);
		hdr.flags = htons(0);
		if (suitable_for_stream_start == NOT_SUITABLE_FOR_STREAM_START) {
			hdr.flags |= htons(METACUBE_FLAGS_NOT_SUITABLE_FOR_STREAM_START);
		}
		hdr.csum = htons(metacube2_compute_crc(&hdr));

		iovec iov;
		iov.iov_base = new char[bytes + sizeof(hdr)];
		iov.iov_len = bytes + sizeof(hdr);

		memcpy(iov.iov_base, &hdr, sizeof(hdr));
		memcpy(reinterpret_cast<char *>(iov.iov_base) + sizeof(hdr), data, bytes);

		queued_data.push_back(iov);
	} else if (encoding == Stream::STREAM_ENCODING_RAW) {
		// Just add the data itself.
		iovec iov;
		iov.iov_base = new char[bytes];
		memcpy(iov.iov_base, data, bytes);
		iov.iov_len = bytes;

		queued_data.push_back(iov);
	} else {
		assert(false);
	}
}

void Stream::process_queued_data()
{
	std::vector<iovec> queued_data_copy;
	int queued_data_last_starting_point_copy = -1;

	// Hold the lock for as short as possible, since add_data_raw() can possibly
	// write to disk, which might disturb the input thread.
	{
		MutexLock lock(&queued_data_mutex);
		if (queued_data.empty()) {
			return;
		}

		swap(queued_data, queued_data_copy);
		swap(queued_data_last_starting_point, queued_data_last_starting_point_copy);
	}

	// Update the last suitable starting point for the stream,
	// if the queued data contains such a starting point.
	assert(queued_data_last_starting_point_copy < ssize_t(queued_data_copy.size()));
	if (queued_data_last_starting_point_copy >= 0) {
		last_suitable_starting_point = bytes_received;
		for (int i = 0; i < queued_data_last_starting_point_copy; ++i) {
			last_suitable_starting_point += queued_data_copy[i].iov_len;
		}
	}

	add_data_raw(queued_data_copy);
	for (size_t i = 0; i < queued_data_copy.size(); ++i) {
		char *data = reinterpret_cast<char *>(queued_data_copy[i].iov_base);
		delete[] data;
	}

	// We have more data, so wake up all clients.
	if (to_process.empty()) {
		swap(sleeping_clients, to_process);
	} else {
		to_process.insert(to_process.end(), sleeping_clients.begin(), sleeping_clients.end());
		sleeping_clients.clear();
	}
}
