/* Copyright (C) 2004 - 2007  db4objects Inc.  http://www.db4o.com */
using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Text;

using Db4objects.Db4o;
using Db4objects.Db4o.Config;
using Db4objects.Db4o.Query;
using Db4objects.Db4o.Linq;

namespace Db4objects.Db4odoc.Performance
{
    class QueryPerformanceBenchmark
    {

        private static int _count = 100000;

        private static int _depth = 3;

        private static bool _isClientServer = false;

        private static bool TCP = true;

        private static string _filePath = "performance.db4o";

        private static string _host = "localhost";

        private static readonly int PORT = 4477;

        private IObjectContainer objectContainer;

        private IObjectServer objectServer;

        private long startTime;

        public static void Main(string[] arguments)
        {
            //new QueryPerformanceBenchmark().RunDifferentQueriesTest();
            //new QueryPerformanceBenchmark().RunDifferentObjectsTest();
            //new QueryPerformanceBenchmark().RunRamDiskTest();
            //new QueryPerformanceBenchmark().RunClientServerTest();
            //new QueryPerformanceBenchmark().RunIndexTest();
            //new QueryPerformanceBenchmark().RunInheritanceTest();
            //new QueryPerformanceBenchmark().RunQueryStructureTest();
            new QueryPerformanceBenchmark().RunQueryAmountOfObjectsTest();
        }

        // end Main

        private void RunQueryAmountOfObjectsTest()
        {
            Init();
            Clean();
            System.Console.WriteLine("Storing " + _count + " of  objects of depth " + _depth);
            Open(Configure());
            Store();
            Close();

            //
            Open(Configure());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            Item item = (Item)query.Execute().Next();
            StopTimer("Select 1 object SODA: " + item._name);
            System.Console.WriteLine("Add some objects of another type and check the query time again:");
		    StoreWithArray();
		    Close();
		    //
		    Open(Configure());
		    StartTimer();
		    query = objectContainer.Query();
		    query.Constrain(typeof(Item));
		    query.Descend("_name").Constrain("level1/1");
		    item = (Item) query.Execute().Next();
		    StopTimer("Select 1 object SODA: " + item._name);
		    Close();
		
		
		    // Add many objects of the same type
            InitLargeDb();
            Clean();
            System.Console.WriteLine("Storing " + _count + " of  objects of depth " + _depth);
            Open(Configure());
            Store();
            Close();

            //
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object SODA: " + item._name);
            Close();
        }
        // end RunQueryAmountOfObjectsTest

        private void RunQueryStructureTest()
        {
            Init();

            Clean();
            System.Console.WriteLine("Storing objects as a bulk:");
            Open(Configure());
            Store();
            Close();

            //
            Open(Configure());
            System.Console.WriteLine("Simple SODA query:");
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            Item item = (Item)query.Execute().Next();
            StopTimer("Select 1 object SODA: " + item._name);
            Close();

            //
            Open(Configure());
            System.Console.WriteLine("Sorted SODA query:");
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").OrderDescending();
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object SODA: " + item._name);
            Close();

            //
            Open(Configure());
            System.Console.WriteLine("SODA query with joins:");
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            IConstraint con = query.Constrain("level2/1");
            query.Descend("_name").OrderDescending().Constrain("level1/1").Or(con);
            IList result = query.Execute();
            StopTimer("Selected " + result.Count + " objects SODA");
            Close();

        }
        // end RunQueryStructureTest

        private void RunDifferentQueriesTest()
        {
            Init();

            Clean();
            System.Console.WriteLine("Storing objects as a bulk:");
            Open(Configure());
            Store();
            Close();

            Open(Configure());
            //
            System.Console.WriteLine("Query by example:");
            StartTimer();
            Item item = (Item)objectContainer.QueryByExample(
                    new Item("level1/1", null)).Next();
            StopTimer("Select 1 object QBE: " + item._name);

            //
            System.Console.WriteLine("SODA:");
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object SODA: " + item._name);

            //
            System.Console.WriteLine("LINQ: ");
            StartTimer();
            var resultLINQ = from Item it in objectContainer
                         where it._name.Equals("level1/1")
                         select it;
            
            IEnumerator<Item> i = resultLINQ.GetEnumerator();
            if (i.MoveNext())
            {
                item = i.Current;
                StopTimer("Select 1 object LINQ: " + item._name);
            }
            
            
            //
            System.Console.WriteLine("Native Query:");
            StartTimer();
            IList<Item> result = objectContainer.Query<Item>(delegate(Item it)
            {
                return it._name.Equals("level1/1");
            });
            item = result[0];
            StopTimer("Select 1 object NQ: " + item._name);
            Close();

            //
            Open(ConfigureUnoptimizedNQ());
            System.Console.WriteLine("Native Query Unoptimized:");
            StartTimer();
            result = objectContainer.Query<Item>(delegate(Item it)
            {
                return it._name.Equals("level1/1");
            });
            item = result[0];
            StopTimer("Select 1 object NQ: " + item._name);

            Close();
        }

        // end RunDifferentQueriesTest


        private void RunRamDiskTest()
        {

            InitForHardDriveTest();
            Clean();
            System.Console.WriteLine("Storing " + _count + " objects of depth " + _depth
                    + " on a hard drive:");
            Open(ConfigureRamDrive());
            Store();
            Close();
            Open(ConfigureRamDrive());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            Item item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();

            InitForRamDriveTest();
            Clean();
            System.Console.WriteLine("Storing " + _count + " objects of depth " + _depth
                    + " on a RAM disk:");
            Open(ConfigureRamDrive());
            Store();
            Close();
            Open(ConfigureRamDrive());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();
        }

        // end RunRamDiskTest

        private void RunClientServerTest()
        {

            InitForClientServer();
            Clean();
            System.Console.WriteLine("Storing " + _count + " objects of depth " + _depth
                    + " remotely:");
            Open(ConfigureClientServer());
            Store();
            Close();
            Open(ConfigureClientServer());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            Item item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();

            Init();
            Clean();
            System.Console.WriteLine("Storing " + _count + " objects of depth " + _depth
                    + " locally:");
            Open(ConfigureClientServer());
            Store();
            Close();
            Open(ConfigureClientServer());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();
        }

        // end RunClientServerTest

        private void RunInheritanceTest()
        {
            Init();
            Clean();
            System.Console.WriteLine("Storing " + _count + " objects of depth " + _depth);
            Open(Configure());
            Store();
            Close();
            Open(Configure());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            Item item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();

            Clean();
            System.Console.WriteLine("Storing " + _count + " inherited objects of depth "
                    + _depth);
            Open(Configure());
            StoreInherited();
            Close();
            Open(Configure());
            StartTimer();
            // Query for item, inheriting objects should be included in the result
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/1");
            item = (Item)query.Execute().Next();
            StopTimer("Select 1 object: " + item._name);
            Close();
        }

        // end RunInheritanceTest

        private void RunDifferentObjectsTest()
        {

            Init();
            System.Console.WriteLine("Storing " + _count + " objects with " + _depth
                    + " levels of embedded objects:");

            Clean();
            System.Console.WriteLine();
            System.Console.WriteLine(" - primitive object with int field");
            Open(Configure());
            StoreSimplest();
            objectContainer.Ext().Purge();
            Close();
            Open(Configure());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(SimplestItem));
            query.Descend("_id").Constrain(1);
            IList result = query.Execute();
            SimplestItem simplestItem = (SimplestItem)result[0];
            StopTimer("Querying SimplestItem: " + simplestItem._id);
            Close();

            Open(Configure());
            System.Console.WriteLine();
            System.Console.WriteLine(" - object with string field");
            Store();
            objectContainer.Ext().Purge();
            Close();
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/2");
            result = query.Execute();
            Item item = (Item)result[0];
            StopTimer("Querying object with string field: " + item._name);
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine();
            System.Console.WriteLine(" - object with StringBuilder field");
            StoreWithStringBuffer();
            objectContainer.Ext().Purge();
            Close();
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(ItemWithStringBuilder));
            query.Descend("_name").Constrain(new StringBuilder("level1/2"));
            result = query.Execute();
            ItemWithStringBuilder itemWithSB = (ItemWithStringBuilder)result[0];
            StopTimer("Querying object with StringBuilder field: "
                    + itemWithSB._name);
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine();
            System.Console.WriteLine(" - object with int array field");
            StoreWithArray();
            objectContainer.Ext().Purge();
            Close();
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(ItemWithArray));
            IQuery idQuery = query.Descend("_id");
            idQuery.Constrain(1);
            idQuery.Constrain(2);
            idQuery.Constrain(3);
            idQuery.Constrain(4);
            result = query.Execute();

            ItemWithArray itemWithArray = (ItemWithArray)result[0];
            StopTimer("Querying object with Array field: [" + itemWithArray._id[0]
                    + ", " + +itemWithArray._id[1] + ", " + +itemWithArray._id[2]
                    + ", " + +itemWithArray._id[0] + "]");
            Close();

            Clean();
            Open(Configure());
            System.Console.WriteLine();
            System.Console.WriteLine(" - object with ArrayList field");
            StoreWithArrayList();
            objectContainer.Ext().Purge();
            Close();
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(ItemWithArrayList));
            query.Descend("_ids").Constrain(1).Contains();
            result = query.Execute();
            ItemWithArrayList itemWithArrayList = (ItemWithArrayList)result[0];
            StopTimer("Querying object with ArrayList field: "
                    + itemWithArrayList._ids.ToString());
            Close();

        }

        // end RunDifferentObjectsTest

        private void RunIndexTest()
        {

            Init();
            System.Console.WriteLine("Storing " + _count + " objects with " + _depth
                    + " levels of embedded objects:");

            Clean();
            System.Console.WriteLine(" - no index");
            Open(Configure());
            Store();
            Close();
            Open(Configure());
            StartTimer();
            IQuery query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/2");
            IList result = query.Execute();
            Item item = (Item)result[0];
            StopTimer("Querying object with string field: " + item._name);
            Close();


            System.Console.WriteLine(" - index on string field");
            // Open to create index
            Open(ConfigureIndex());
            Close();
            Open(Configure());
            StartTimer();
            query = objectContainer.Query();
            query.Constrain(typeof(Item));
            query.Descend("_name").Constrain("level1/2");
            result = query.Execute();
            item = (Item)result[0];
            StopTimer("Querying object with string field: " + item._name);
            Close();
        }

        // end RunIndexTest

        private void Init()
        {
            _filePath = "performance.db4o";
            // amount of objects
            _count = 10000;
            // depth of objects
            _depth = 3;
            _isClientServer = false;

        }

        // end Init

        private void InitLargeDb()
        {
            _filePath = "performance.db4o";
            _count = 100000;
            _depth = 3;
            _isClientServer = false;

        }

        // end InitLargeDb


        private void InitForClientServer()
        {
            _filePath = "performance.db4o";
            _isClientServer = true;
            _host = "localhost";
        }

        // end InitForClientServer

        private void InitForRamDriveTest()
        {
            _count = 30000;
            _depth = 3;
            _filePath = "r:\\performance.db4o";
            _isClientServer = false;

        }

        // end InitForRamDriveTest

        private void InitForHardDriveTest()
        {
            _count = 10000;
            _depth = 3;
            _filePath = "performance.db4o";
            _isClientServer = false;

        }

        // end InitForHardDriveTest


        private void Clean()
        {
            File.Delete(_filePath);
        }

        // end Clean

        private IConfiguration Configure()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            return config;
        }

        // end Configure

        private IConfiguration ConfigureUnoptimizedNQ()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.OptimizeNativeQueries(false);
            return config;
        }
        // end ConfigureUnoptimizedNQ

        private IConfiguration ConfigureIndex()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.ObjectClass(typeof(Item)).ObjectField("_name").Indexed(true);
            return config;
        }

        // end ConfigureIndex

        private IConfiguration ConfigureClientServer()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.Queries().EvaluationMode(QueryEvaluationMode.Immediate);
            config.ClientServer().SingleThreadedClient(true);
            return config;
        }

        // end ConfigureClientServer

        private IConfiguration ConfigureRamDrive()
        {
            IConfiguration config = Db4oFactory.NewConfiguration();
            config.FlushFileBuffers(true);
            return config;
        }

        // end ConfigureRamDrive

        private void Store()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                Item item = new Item("level" + i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new Item("level" + i + "/" + j, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end Store

        private void StoreInherited()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                ItemDerived item = new ItemDerived("level" + i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new ItemDerived("level" + i + "/" + j, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end StoreInherited

        private void StoreWithStringBuffer()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                ItemWithStringBuilder item = new ItemWithStringBuilder(
                        new StringBuilder("level" + i), null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new ItemWithStringBuilder(new StringBuilder("level" + i
                            + "/" + j), item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end StoreWithStringBuffer

        private void StoreSimplest()
        {
            StartTimer();
            for (int i = 0; i < _count; i++)
            {
                SimplestItem item = new SimplestItem(i, null);
                for (int j = 1; j < _depth; j++)
                {
                    item = new SimplestItem(i, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end StoreSimplest

        private void StoreWithArray()
        {
            StartTimer();
            int[] array = new int[] { 1, 2, 3, 4 };
            for (int i = 0; i < _count; i++)
            {
                int[] id = new int[] { 1, 2, 3, 4 };
                ItemWithArray item = new ItemWithArray(id, null);
                for (int j = 1; j < _depth; j++)
                {
                    int[] id1 = new int[] { 1, 2, 3, 4 };
                    item = new ItemWithArray(id1, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end StoreWithArray

        private void StoreWithArrayList()
        {
            StartTimer();
            ArrayList idList = new ArrayList();
            idList.Add(1);
            idList.Add(2);
            idList.Add(3);
            idList.Add(4);
            for (int i = 0; i < _count; i++)
            {
                ArrayList ids = new ArrayList();
                ids.AddRange(idList);
                ItemWithArrayList item = new ItemWithArrayList(ids, null);
                for (int j = 1; j < _depth; j++)
                {
                    ArrayList ids1 = new ArrayList();
                    ids1.AddRange(idList);
                    item = new ItemWithArrayList(ids1, item);
                }
                objectContainer.Store(item);
            }
            objectContainer.Commit();
            StopTimer("Store " + TotalObjects() + " objects");
        }

        // end StoreWithArrayList

        private int TotalObjects()
        {
            return _count * _depth;
        }

        // end TotalObjects

        private void Open(IConfiguration Configure)
        {
            if (_isClientServer)
            {
                int port = TCP ? PORT : 0;
                string user = "db4o";
                string password = user;
                objectServer = Db4oFactory.OpenServer(Configure, _filePath, port);
                objectServer.GrantAccess(user, password);
                objectContainer = TCP ? Db4oFactory
                        .OpenClient(Configure, _host, port, user, password) : objectServer
                        .OpenClient(Configure);
            }
            else
            {
                objectContainer = Db4oFactory.OpenFile(Configure, _filePath);
            }
        }

        // end Open

        private void Close()
        {
            objectContainer.Close();
            if (_isClientServer)
            {
                objectServer.Close();
            }
        }

        // end Close

        private void StartTimer()
        {
            startTime = DateTime.Now.Ticks;
        }

        // end StartTimer

        private void StopTimer(string message)
        {
            long stop = DateTime.Now.Ticks;
            TimeSpan ts = new TimeSpan(stop - startTime);
            System.Console.WriteLine(message + ": " + ts.TotalMilliseconds + " ms");
        }

        // end StopTimer

        public class Item
        {

            public string _name;
            public Item _child;

            public Item()
            {

            }

            public Item(string name, Item child)
            {
                _name = name;
                _child = child;
            }
        }

        // end Item

        public class ItemDerived : Item
        {

            public ItemDerived(string name, ItemDerived child)
                : base(name, child)
            {
            }
        }

        // end ItemDerived

        public class ItemWithStringBuilder
        {

            public StringBuilder _name;
            public ItemWithStringBuilder _child;

            public ItemWithStringBuilder()
            {
            }

            public ItemWithStringBuilder(StringBuilder name,
                    ItemWithStringBuilder child)
            {
                _name = name;
                _child = child;
            }
        }

        // end ItemWithStringBuilder

        public class SimplestItem
        {

            public int _id;
            public SimplestItem _child;

            public SimplestItem()
            {
            }

            public SimplestItem(int id, SimplestItem child)
            {
                _id = id;
                _child = child;
            }
        }

        // end SimplestItem

        public class ItemWithArray
        {

            public int[] _id;
            public ItemWithArray _child;

            public ItemWithArray()
            {
            }

            public ItemWithArray(int[] id, ItemWithArray child)
            {
                _id = id;
                _child = child;
            }
        }

        // end ItemWithArray

        public class ItemWithArrayList
        {

            public ArrayList _ids;
            public ItemWithArrayList _child;

            public ItemWithArrayList()
            {
            }

            public ItemWithArrayList(ArrayList ids, ItemWithArrayList child)
            {
                _ids = ids;
                _child = child;
            }
        }
        // end ItemWithArrayList


    }
}
