from typing import Tuple, TYPE_CHECKING

if TYPE_CHECKING:
    import lsprotocol.types as types
    from debputy.lsp.debputy_ls import DebputyLanguageServer
else:
    import debputy.lsprotocol.types as types

    try:
        from debputy.lsp.debputy_ls import DebputyLanguageServer
    except ImportError:
        DebputyLanguageServer = type


def _locate_cursor(text: str) -> Tuple[str, types.Position]:
    lines = text.splitlines(keepends=True)
    for line_no in range(len(lines)):
        line = lines[line_no]
        try:
            c = line.index("<CURSOR>")
        except ValueError:
            continue
        line = line.replace("<CURSOR>", "")
        lines[line_no] = line
        pos = types.Position(line_no, c)
        return "".join(lines), pos
    raise ValueError('Missing "<CURSOR>" marker')


def put_doc_with_cursor(
    ls: "DebputyLanguageServer",
    uri: str,
    language_id: str,
    content: str,
) -> types.Position:
    cleaned_content, cursor_pos = _locate_cursor(content)
    put_doc_no_cursor(
        ls,
        uri,
        language_id,
        cleaned_content,
    )
    return cursor_pos


def put_doc_no_cursor(
    ls: "DebputyLanguageServer",
    uri: str,
    language_id: str,
    content: str,
) -> None:
    doc_version = 1
    existing = ls.workspace.text_documents.get(uri)
    if existing is not None:
        doc_version = existing.version + 1
    ls.workspace.put_text_document(
        types.TextDocumentItem(
            uri,
            language_id,
            doc_version,
            content,
        )
    )
