/*
 * Copyright 2020 Advanced Micro Devices, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER(S) OR AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR
 * OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 */

#include "igt.h"
#include "igt_sysfs.h"
#include "igt_amd.h"
#include <fcntl.h>

IGT_TEST_DESCRIPTION("Test EDID parsing and debugfs reporting on Freesync displays");

/* Maximumm pipes on any AMD ASIC. */
#define MAX_PIPES 6
#define EDID_SIZE 256
#define EDID_PATH "/sys/class/drm/card0-%s/edid"

/* Common test data. */
struct vrr_range {
	unsigned int min;
	unsigned int max;
};

typedef struct data {
	igt_display_t display;
	igt_plane_t *primary;
	igt_output_t *output[MAX_PIPES];
	int fd;
	struct vrr_range expected_range;
} data_t;

/* Test flags. */
enum {
	TEST_NONE = 1 << 0,
	TEST_SUSPEND = 1 << 1,
};

struct {
	const char *name;
	uint32_t connector_type;
	const unsigned char edid[256];
	const struct vrr_range range;
} edid_database[] = {
	{
		/* EDID Version 1.4. Timing requires 2 DP lanes. */
		"External DP",
		DRM_MODE_CONNECTOR_DisplayPort,
		{
		0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
		0x06, 0xb3, 0xaf, 0x24, 0x01, 0x01, 0x01, 0x01,
		0x00, 0x1d, 0x01, 0x04, 0x80, 0x35, 0x1e, 0x78,
		0x2b, 0x51, 0xb5, 0xa4, 0x54, 0x4f, 0xa0, 0x26,
		0x0d, 0x50, 0x54, 0xbf, 0xcf, 0x00, 0x81, 0x40,
		0x81, 0x80, 0x95, 0x00, 0x71, 0x4f, 0x81, 0xc0,
		0xb3, 0x00, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a,
		0x80, 0x18, 0x71, 0x38, 0x2d, 0x40, 0x58, 0x2c,
		0x45, 0x00, 0x0f, 0x28, 0x21, 0x00, 0x00, 0x1e,
		0xfc, 0x7e, 0x80, 0x88, 0x70, 0x38, 0x12, 0x40,
		0x18, 0x20, 0x35, 0x00, 0x0f, 0x28, 0x21, 0x00,
		0x00, 0x1e, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x30,
		0x90, 0x1e, 0xb4, 0x22, 0x01, 0x0a, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfc,
		0x00, 0x41, 0x53, 0x55, 0x53, 0x20, 0x56, 0x50,
		0x32, 0x34, 0x39, 0x0a, 0x20, 0x20, 0x01, 0x91,
		0x02, 0x03, 0x2d, 0xf1, 0x4f, 0x01, 0x03, 0x04,
		0x13, 0x1f, 0x12, 0x02, 0x11, 0x90, 0x0e, 0x0f,
		0x1d, 0x1e, 0x3f, 0x40, 0x23, 0x09, 0x07, 0x07,
		0x83, 0x01, 0x00, 0x00, 0x67, 0x03, 0x0c, 0x00,
		0x10, 0x00, 0x00, 0x44, 0x68, 0x1a, 0x00, 0x00,
		0x01, 0x01, 0x30, 0x90, 0xe6, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16
		},
		{48, 144},
	},

	{
		/* HDMI EDID from ASUS VP249QGR */
		"ASUS VP249QGR HDMI",
		DRM_MODE_CONNECTOR_HDMIA,
		{
		0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
		0x06, 0xb3, 0xaf, 0x24, 0x01, 0x01, 0x01, 0x01,
		0x00, 0x1d, 0x01, 0x03, 0x80, 0x35, 0x1e, 0x78,
		0x2a, 0x51, 0xb5, 0xa4, 0x54, 0x4f, 0xa0, 0x26,
		0x0d, 0x50, 0x54, 0xbf, 0xcf, 0x00, 0x81, 0x40,
		0x81, 0x80, 0x95, 0x00, 0x71, 0x4f, 0x81, 0xc0,
		0xb3, 0x00, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a,
		0x80, 0x18, 0x71, 0x38, 0x2d, 0x40, 0x58, 0x2c,
		0x45, 0x00, 0x0f, 0x28, 0x21, 0x00, 0x00, 0x1e,
		0xfc, 0x7e, 0x80, 0x88, 0x70, 0x38, 0x12, 0x40,
		0x18, 0x20, 0x35, 0x00, 0x0f, 0x28, 0x21, 0x00,
		0x00, 0x1e, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x30,
		0x90, 0x1e, 0xb4, 0x22, 0x00, 0x0a, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfc,
		0x00, 0x41, 0x53, 0x55, 0x53, 0x20, 0x56, 0x50,
		0x32, 0x34, 0x39, 0x0a, 0x20, 0x20, 0x01, 0x94,
		0x02, 0x03, 0x2d, 0xf1, 0x4f, 0x01, 0x03, 0x04,
		0x13, 0x1f, 0x12, 0x02, 0x11, 0x90, 0x0e, 0x0f,
		0x1d, 0x1e, 0x3f, 0x40, 0x23, 0x09, 0x07, 0x07,
		0x83, 0x01, 0x00, 0x00, 0x67, 0x03, 0x0c, 0x00,
		0x10, 0x00, 0x00, 0x44, 0x68, 0x1a, 0x00, 0x00,
		0x01, 0x01, 0x30, 0x90, 0xe6, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16
		},
		{48, 144},
	},

	{
		/* EDID Version 1.4. Timing requires 2 DP lanes. */
		"eDP",
		DRM_MODE_CONNECTOR_eDP,
		{
		0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,
		0x06, 0xb3, 0xaf, 0x24, 0x01, 0x01, 0x01, 0x01,
		0x00, 0x1d, 0x01, 0x04, 0x80, 0x35, 0x1e, 0x78,
		0x2b, 0x51, 0xb5, 0xa4, 0x54, 0x4f, 0xa0, 0x26,
		0x0d, 0x50, 0x54, 0xbf, 0xcf, 0x00, 0x81, 0x40,
		0x81, 0x80, 0x95, 0x00, 0x71, 0x4f, 0x81, 0xc0,
		0xb3, 0x00, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a,
		0x80, 0x18, 0x71, 0x38, 0x2d, 0x40, 0x58, 0x2c,
		0x45, 0x00, 0x0f, 0x28, 0x21, 0x00, 0x00, 0x1e,
		0xfc, 0x7e, 0x80, 0x88, 0x70, 0x38, 0x12, 0x40,
		0x18, 0x20, 0x35, 0x00, 0x0f, 0x28, 0x21, 0x00,
		0x00, 0x1e, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x30,
		0x90, 0x1e, 0xb4, 0x22, 0x01, 0x0a, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfc,
		0x00, 0x41, 0x53, 0x55, 0x53, 0x20, 0x56, 0x50,
		0x32, 0x34, 0x39, 0x0a, 0x20, 0x20, 0x01, 0x91,
		0x02, 0x03, 0x2d, 0xf1, 0x4f, 0x01, 0x03, 0x04,
		0x13, 0x1f, 0x12, 0x02, 0x11, 0x90, 0x0e, 0x0f,
		0x1d, 0x1e, 0x3f, 0x40, 0x23, 0x09, 0x07, 0x07,
		0x83, 0x01, 0x00, 0x00, 0x67, 0x03, 0x0c, 0x00,
		0x10, 0x00, 0x00, 0x44, 0x68, 0x1a, 0x00, 0x00,
		0x01, 0x01, 0x30, 0x90, 0xe6, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16
		},
		{48, 144},
	},
};

static void test_init(data_t *data)
{
	igt_display_t *display = &data->display;
	int i;

	for_each_pipe(display, i) {
		igt_output_t *output = &display->outputs[i];

		data->output[i] = output;
	}

	igt_display_reset(display);
}

static void test_fini(data_t *data)
{
	igt_display_reset(&data->display);
}

static int find_test_edid_index(uint32_t connector_type)
{
	int i;

	for(i = 0; i < sizeof(edid_database)/sizeof(edid_database[0]); ++i) {
		if (edid_database[i].connector_type == connector_type) {
			return i;
		}
	}

	return -1;
}

/* Returns the min and max vrr range from the connector debugfs. */
static struct vrr_range get_freesync_range(data_t *data, igt_output_t *output)
{
	char buf[256];
	char *start_loc;
	int fd, res;
	struct vrr_range range;

	fd = igt_debugfs_connector_dir(data->fd, output->name, O_RDONLY);
	igt_assert(fd >= 0);

	/* example usage: cat /sys/kernel/debug/dri/0/DP-1/vrr_range */
	res = igt_debugfs_simple_read(fd, "vrr_range", buf, sizeof(buf));
	igt_require(res > 0);

	close(fd);

	igt_assert(start_loc = strstr(buf, "Min: "));
	igt_assert_eq(sscanf(start_loc, "Min: %u", &range.min), 1);

	igt_assert(start_loc = strstr(buf, "Max: "));
	igt_assert_eq(sscanf(start_loc, "Max: %u", &range.max), 1);

	return range;
}

static void trigger_edid_parse(data_t *data, igt_output_t *output, uint32_t test_flags)
{
	if (test_flags & TEST_SUSPEND)
		igt_system_suspend_autoresume(SUSPEND_STATE_MEM,
					      SUSPEND_TEST_NONE);
	else
		igt_amd_trigger_hotplug(data->fd, output->name);

	/* more safe margin until resume and hotplug is completed */
	usleep(1500000);
}

/* Returns true if an output supports VRR. */
static bool has_vrr(igt_output_t *output)
{
	return igt_output_has_prop(output, IGT_CONNECTOR_VRR_CAPABLE) &&
	       igt_output_get_prop(output, IGT_CONNECTOR_VRR_CAPABLE);
}

static void parse_vrr_gange_from_edid(data_t *data, uint8_t *edid, int index)
{
	bool max_rate_offset = false;
	bool min_rate_offset = false;

	/* Check Bytes 4 Vertical rate offsets
	 * Vertical rate offsets:
	 * 00 = none;
	 * 10 = +255 Hz for max. rate;
	 * 11 = +255 Hz for max. and min. rates.
	 */
	if ((edid[index + 4] & 0b10) == 0b10)
		max_rate_offset = true;
	else if ((edid[index + 4] & 0b11) == 0b11) {
		max_rate_offset = true;
		min_rate_offset = true;
	}

	/* Bytes 5 Min vertical field rate (1–255 Hz; 256–510 Hz, if offset).*/
	data->expected_range.min =
		min_rate_offset ? edid[index + 5] + 255 : edid[index + 5];
	/* Bytes 6 Max vertical field rate (1–255 Hz; 256–510 Hz, if offset).*/
	data->expected_range.max =
		max_rate_offset ? edid[index + 6] + 255 : edid[index + 6];
}

static bool find_vrr_range_from_edid(data_t *data, igt_output_t *output)
{
	char edid_path[PATH_MAX] = "\0";
	uint8_t sink_edid[EDID_SIZE] = "\0";
	const uint8_t range_limits_head[4] = {0x00, 0x00, 0x00, 0xFD};
	const unsigned int range_head_size = sizeof(range_limits_head);
	int fd, i, read_size, index = 0;

	/* Get EDID */
	igt_assert(snprintf(edid_path, PATH_MAX, EDID_PATH,
			   output->name) < PATH_MAX);

	fd = open(edid_path, O_RDONLY);
	if (fd == -1)
		return false;

	read_size = read(fd, sink_edid, EDID_SIZE);
	close(fd);
	if (read_size < 0)
		return false;

	/* Find Display Range Limits Descriptor block */
	while (index < EDID_SIZE - range_head_size) {
		for (i = 0; i < range_head_size; i++) {
			if (sink_edid[index+i] != range_limits_head[i])
				break;
			else if (i == range_head_size-1) {
				/* Found Display Range Limits Descriptor block */
				parse_vrr_gange_from_edid(data, sink_edid, index);
				return true;
			}
		}
		index++;
	}

	return false;
}

/* Check if EDID parsing is correctly reporting Freesync capability
 * by overriding EDID with ones from golden sample.
 */
static void test_freesync_parsing_base(data_t *data, uint32_t test_flags)
{
	const struct edid *edid;
	struct vrr_range range, expected_range;
	igt_output_t *output;
	int j, test_conn_cnt = 0;
	igt_display_t *display = &data->display;

	test_init(data);

	igt_amd_require_hpd(&data->display, data->fd);

	for_each_connected_output(display, output) {
		/* find a test EDID */
		j = find_test_edid_index(output->config.connector->connector_type);
		if (j == -1)
			continue;

		edid = (const struct edid *)edid_database[j].edid;
		expected_range = edid_database[j].range;

		if (has_vrr(output)) {
			/* A VRR sink, just parsing range from EDID directly */

			trigger_edid_parse(data, output, test_flags);

			igt_assert_f(find_vrr_range_from_edid(data, output),
				"Cannot parsing VRR range from EDID\n");

			expected_range.min = data->expected_range.min;
			expected_range.max = data->expected_range.max;
			range = get_freesync_range(data, output);
		} else {
			/* A non-VRR sink. Override a golden EDID */
			/* eDP allow read edid for each display detection */
			if (output->config.connector->connector_type == DRM_MODE_CONNECTOR_eDP)
				igt_amd_allow_edp_hotplug_detect(data->fd, output->name, true);

			/* force to use hard coded VRR EDID */
			kmstest_force_edid(data->fd, output->config.connector, edid);

			trigger_edid_parse(data, output, test_flags);

			range = get_freesync_range(data, output);

			/* undo EDID override. re-parse EDID of display */
			kmstest_force_edid(data->fd, output->config.connector, NULL);
			igt_amd_trigger_hotplug(data->fd, output->name);

			/* eDP dis-allow read edid for each display detection */
			if (output->config.connector->connector_type == DRM_MODE_CONNECTOR_eDP)
				igt_amd_allow_edp_hotplug_detect(data->fd, output->name, false);
		}

		test_conn_cnt++;

		igt_assert_f(range.min == expected_range.min &&
				range.max == expected_range.max,
				"Expecting Freesync range %d-%d, got %d-%d\n",
				expected_range.min, expected_range.max,
				range.min, range.max);
		igt_info("%s Freesync range: %d-%d\n", output->name, range.min, range.max);
	}

	test_fini(data);
	igt_skip_on(test_conn_cnt == 0);
}

static inline void test_freesync_parsing(data_t *data)
{
	test_freesync_parsing_base(data, TEST_NONE);
}

static inline void test_freesync_parsing_suspend(data_t *data)
{
	test_freesync_parsing_base(data, TEST_SUSPEND);
}

/* More relaxed checking on Freesync capability.
 * Only checks if frame rate range is within legal range.
 * Display under test MUST be VRR capable.
 */
static void test_freesync_range_base(data_t *data, uint32_t test_flags)
{
	struct vrr_range range;
	igt_output_t *output;
	int test_conn_cnt = 0;
	igt_display_t *display = &data->display;

	test_init(data);

	igt_amd_require_hpd(&data->display, data->fd);

	for_each_connected_output(display, output) {
		igt_debug_on_f(!has_vrr(output), "Requires output supports VRR\n");

		if (!has_vrr(output)) {
			igt_info("connector %s is not VRR capable\n", output->name);
			continue;
		}

		trigger_edid_parse(data, output, test_flags);

		range = get_freesync_range(data, output);

		igt_assert_f(range.min != 0 &&
				range.max != 0 &&
				range.max - range.min > 10,
				"Invalid Freesync range %d-%d\n",
				range.min, range.max);
		igt_info("%s Freesync range: %d-%d\n", output->name, range.min, range.max);

		test_conn_cnt++;
	}

	test_fini(data);
	igt_skip_on(test_conn_cnt == 0);
}

static inline void test_freesync_range(data_t *data)
{
	test_freesync_range_base(data, TEST_NONE);
}

static inline void test_freesync_range_suspend(data_t *data)
{
	test_freesync_range_base(data, TEST_SUSPEND);
}

igt_main
{
	data_t data;

	igt_skip_on_simulation();

	memset(&data, 0, sizeof(data));

	igt_fixture
	{
		data.fd = drm_open_driver_master(DRIVER_AMDGPU);

		kmstest_set_vt_graphics_mode();

		igt_display_require(&data.display, data.fd);
		igt_require(data.display.is_atomic);
		igt_display_require_output(&data.display);
	}

	igt_describe("Hard coded Freesync EDID parsing");
	igt_subtest("freesync-parsing") test_freesync_parsing(&data);

	igt_describe("Hard coded Freesync EDID parsing after suspend");
	igt_subtest("freesync-parsing-suspend") test_freesync_parsing_suspend(&data);

	igt_describe("Freesync range from display");
	igt_subtest("freesync-range") test_freesync_range(&data);

	igt_describe("Freesync range from display after suspend");
	igt_subtest("freesync-range-suspend") test_freesync_range_suspend(&data);

	igt_fixture
	{
		igt_display_fini(&data.display);
		drm_close_driver(data.fd);
	}
}
