package latexDraw.parsers.svg.elements.path;

import java.awt.geom.Arc2D;

/**
 * Defines the SVGPath arc segment.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 10/20/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 * @since 0.1
 */
public class SVGPathSegArc extends SVGPathSeg
{
	/** The X-coordinate of the second point of the arc. @since 0.1 */
	protected double x;
	
	/** The Y-coordinate of the second point of the arc. @since 0.1 */
	protected double y;
	
	/** The x radius of the arc. @since 0.1 */
	protected double rx;
	
	/** The y radius of the arc. @since 0.1 */
	protected double ry;
	
	/** The x-axis rotation angle. @since 0.1 */
	protected double angle;
	
	protected boolean largeArcFlag;
	
	protected boolean sweepFlag;

	
	/**
	 * The main constructor.
	 * @param x The X-coordinate of the second point of the arc.
	 * @param y The Y-coordinate of the second point of the arc.
	 * @param rx The x radius of the arc.
	 * @param ry The y radius of the arc.
	 * @param angle The x-axis rotation angle.
	 * @param isRelative True: the path segment is relative, false it is absolute.
	 */
	public SVGPathSegArc(double x, double y, double rx, double ry, double angle, 
						boolean largeArcFlag, boolean sweepFlag, boolean isRelative)
	{
		super(isRelative);
		
		this.x = x;
		this.y = y;
		this.rx = rx;
		this.ry = ry;
		this.angle = angle;
		this.largeArcFlag = largeArcFlag;
		this.sweepFlag = sweepFlag;
	}
	
	
	
	/**
	 * Creates a Java Arc2D corresponding to the position and the angles of the arc segment (computations based on the SVG 
	 * specification instructions concerning the build of an arc, p. 643-649).
	 * @param x0 The X-coordinate of the initial position.
	 * @param y0 The Y-coordinate of the initial position.
	 * @return An Java Arc2D with double values.
	 * @since 0.1
	 */
    public Arc2D getArc2D(double x0, double y0)
	{
    	double a 	= getAngle();
    	double _rx 	= getRX();
    	double _ry 	= getRY();
    	double _x 	= getX();
    	double _y	= getY();
    	boolean laf	= isLargeArcFlag();
    	boolean sf	= isSweepFlag();
    	
		double dx2 = (x0-_x)/2.;
		double dy2 = (y0-_y)/2.;
		a = Math.toRadians(a%360.);

		// Step 1: Compute (x1', y1')
		double x1 = Math.cos(a)*dx2 + Math.sin(a)*dy2;
		double y1 = -Math.sin(a)*dx2 + Math.cos(a)*dy2;
		
		// Ensure radii are large enough
		_rx = Math.abs(_rx);
		_ry = Math.abs(_ry);
		double Prx = _rx*_rx;
		double Pry = _ry*_ry;
		double Px1 = x1*x1;
		double Py1 = y1*y1;
		double radiiCheck = Px1/Prx + Py1/Pry;
		
		if(radiiCheck>1)
		{
			_rx = Math.sqrt(radiiCheck) * _rx;
			_ry = Math.sqrt(radiiCheck) * _ry;
			Prx = _rx * _rx;
			Pry = _ry * _ry;
		}

		// Step 2: Compute (cx1, cy1)
		double sign = (laf == sf) ? -1 : 1;
		double sq = ((Prx * Pry) - (Prx * Py1) - (Pry * Px1)) / ((Prx * Py1) + (Pry * Px1));
		sq = (sq < 0) ? 0 : sq;
		double coef = (sign * Math.sqrt(sq));
		double cx1 = coef * ((_rx * y1) / _ry);
		double cy1 = coef * -((_ry * x1) / _rx);

		// Step 3: Compute (cx, cy) from (cx1, cy1)
		double sx2 = (x0+_x)/2.;
		double sy2 = (y0+_y)/2.;
		double cx = sx2 + (Math.cos(a) * cx1 - Math.sin(a) * cy1);
		double cy = sy2 + (Math.sin(a) * cx1 + Math.cos(a) * cy1);

		// Step 4: Compute the angleStart (angle1) and the angleExtent (dangle)
		double ux = (x1 - cx1) / _rx;
		double uy = (y1 - cy1) / _ry;
		double vx = (-x1 - cx1) / _rx;
		double vy = (-y1 - cy1) / _ry;
		double p = ux, n = Math.sqrt((ux * ux) + (uy * uy));

		sign = (uy < 0) ? -1. : 1.;
		double angleStart = Math.toDegrees(sign * Math.acos(p / n));

		// Compute the angle extent
		n = Math.sqrt((ux * ux + uy * uy) * (vx * vx + vy * vy));
		p = ux * vx + uy * vy;
		sign = (ux * vy - uy * vx < 0) ? -1. : 1.;
		
		double angleExtent = Math.toDegrees(sign * Math.acos(p / n));
		
		if(!sf && angleExtent > 0)
			angleExtent -= 360.;
		else
			if(sf && angleExtent < 0)
				angleExtent += 360.;
		
		return new Arc2D.Double(cx-_rx, cy-_ry, _rx*2., _ry*2., -angleStart%360., -angleExtent%360., Arc2D.OPEN);
	}
	
	
	
	/**
	 * @return the x.
	 * @since 0.1
	 */
	public double getX()
	{
		return x;
	}

	/**
	 * @return the y.
	 * @since 0.1
	 */
	public double getY()
	{
		return y;
	}

	/**
	 * @return the rx.
	 * @since 0.1
	 */
	public double getRX()
	{
		return rx;
	}

	/**
	 * @return the ry.
	 * @since 0.1
	 */
	public double getRY()
	{
		return ry;
	}

	/**
	 * @return the angle.
	 * @since 0.1
	 */
	public double getAngle()
	{
		return angle;
	}

	/**
	 * @return the largeArcFlag.
	 * @since 0.1
	 */
	public boolean isLargeArcFlag()
	{
		return largeArcFlag;
	}

	/**
	 * @return the sweepFlag.
	 * @since 0.1
	 */
	public boolean isSweepFlag()
	{
		return sweepFlag;
	}

	
	@Override
	public PathSeg getType()
	{
		return isRelative() ? PathSeg.ARC_REL : PathSeg.ARC_ABS;
	}
	
	
	@Override
	public String toString()
	{
		StringBuffer stringBuffer = new StringBuffer();
		
		stringBuffer.append((isRelative() ? "a" : "A")); //$NON-NLS-1$ //$NON-NLS-2$
		stringBuffer.append(' ');
		stringBuffer.append(rx);
		stringBuffer.append(' ');
		stringBuffer.append(ry);
		stringBuffer.append(' ');
		stringBuffer.append(angle);
		stringBuffer.append(' ');
		stringBuffer.append((largeArcFlag?'1':'0'));
		stringBuffer.append(' ');
		stringBuffer.append((sweepFlag?'1':'0'));
		stringBuffer.append(' ');
		stringBuffer.append(x);
		stringBuffer.append(' ');
		stringBuffer.append(y);
		
		return stringBuffer.toString();
	}
}
