package tests::ReportParserHTMLDocBookFormatterTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::ReportParser::HTMLDocBookFormatter qw/ dbk2html /;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}

sub _element_is_normalized {
    my ( $self, $element ) = @_;

    $self->annotate( "_element_is_normalized: $element\n" );
    my $dbk_text = "<para>Wawa <$element>bobo</$element> zozo</para>";
    my $e_text = <<EOD;
    <div class="para">Wawa <span class="$element">bobo</span> zozo</div>
EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text, 200 ) );

    return;
}

sub _test_admonition {
    my ( $self, $admonition ) = @_;

    $self->annotate( "testing admonition '$admonition'\n" );
    my $dbk_text = "<$admonition><para>test</para></$admonition>";
    my $uc_admonition = ucfirst( $admonition );
    my $e_text = <<_EOD;
    <div class="$admonition"><span class="admonition">$uc_admonition:</span>
      <div class="para">test</div>
    </div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    $dbk_text = "<$admonition><para>paragraph 1</para><para>paragraph 2</para></$admonition>";
    $e_text = <<_EOD;
    <div class="$admonition"><span class="admonition">$uc_admonition:</span>
      <div class="para">paragraph 1</div>
      <div class="para">paragraph 2</div>
    </div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

sub _test_autopcdata {
    my ( $self, $data ) = @_;

    $self->annotate( "testing autopcdata '$data'\n" );
    $self->_element_is_normalized( $data );

    return;
}

sub test_admonitions {
    my $self = $_[0];

    my @admonitions = ( 'note', 'tip', 'important', 'caution', 'warning' );
    foreach my $admonition ( @admonitions ) {
        $self->_test_admonition( $admonition );
    }

    return;
}

sub test_all_autopcdata {
    my $self = $_[0];

    foreach my $data ( qw/ phrase trademark wordasword action application
                           command database errorcode errortype filename
                           hardware interface keycap keycode keycombo keysym
                           literal markup replaceable returnvalue sgmltag
                           systemitem token productname productnumber / )
    {
        $self->_test_autopcdata( $data );
    }

    return;
}

sub test_orderedlist {
    my $self = $_[0];

    my $dbk_text = '<orderedlist><listitem><para>test1</para></listitem><listitem><para>test2</para></listitem></orderedlist>';
    my $e_text = <<_EOD;
    <ol>
      <li>
\t<div class="para">test1</div>
      </li>
      <li>
\t<div class="para">test2</div>
      </li>
    </ol>
_EOD
     $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

     return;
}

sub test_itemizedlist {
    my $self = $_[0];

    my $dbk_text = '<itemizedlist><listitem><para>test1</para></listitem><listitem><para>test2</para></listitem></itemizedlist>';
    my $e_text = <<_EOD;
    <ul>
      <li>
\t<div class="para">test1</div>
      </li>
      <li>
\t<div class="para">test2</div>
      </li>
    </ul>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

# varlistentry and listitem are tested here too
sub test_variablelist {
    my $self = $_[0];

    my $dbk_text = <<_EOD;
    <variablelist>
      <title>This is a title for our list</title>
      <para>A paragraph before the actual list</para>
      <varlistentry>
        <term>term 1</term>
        <listitem>
          <para>definition 1</para>
        </listitem>
      </varlistentry>
      <varlistentry>
        <term>wawa term</term>
        <listitem>
          <para>wawa super text</para>
        </listitem>
      </varlistentry>
    </variablelist>
_EOD
    my $e_text = <<_EOD;
    <div class="variablelist">
      <span class="title">This is a title for our list</span>
      <div class="para">A paragraph before the actual list</div>
      <dl>
\t<dt>term 1</dt>
\t<dd>
\t  <div class="para">definition 1</div>
\t</dd>
\t<dt>wawa term</dt>
\t<dd>
\t  <div class="para">wawa super text</div>
\t</dd>
      </dl>
    </div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

sub test_para {
    my $self = $_[0];

    my $dbk_text = '<para>This is a very very very very very very very very very very very very very very very long paragraph</para>';
    my $e_text = <<_EOD;
    <div class="para">This is a very very very very very very very very very
    very very very very very very long paragraph</div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

sub test_ulink {
    my $self = $_[0];

    my $dbk_text = '<para>Go to <ulink url="http://www.logreport.org/">The LogReport Foundation</ulink> website.</para>';
    my $e_text = <<_EOD;
    <div class="para">Go to <a href="http://www.logreport.org/">The LogReport
    Foundation</a> website.</div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

sub test_email {
    my $self = $_[0];

    my $dbk_text = '<para><email>Wolfgang@logreport.org</email></para>';
    my $e_text = <<_EOD;
    <div class="para"><a href="mailto:Wolfgang\@logreport.org">Wolfgang\@logreport.org</a></div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text, 200 ) );

    return;
}

sub _test_variable {
    my ( $self, $vartype ) = @_;

    my $dbk_text = "<para><$vartype>TERM</$vartype></para>";
    my $e_text = <<_EOD;
    <div class="para"><var class="$vartype">TERM</var></div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text, 200 ) );

    return;
}

sub test_variables {
    my $self = $_[0];

    foreach my $vartype ( qw/ varname envar constant structfield structname function
                              option optional parameter classname symbol property
                              type errorname / )
    {
        $self->annotate( "testing variable type '$vartype'\n" );
        $self->_test_variable( $vartype );
    }
}

sub test_quote {
    my $self = $_[0];

    my $dbk_text = '<para><quote>Linux</quote></para>';
    my $e_text = <<_EOD;
    <div class="para"><q>Linux</q></div>
_EOD
    $self->assert_str_equals( $e_text, dbk2html( $dbk_text ) );

    return;
}

1;
