#include "osl/misc/ncores.h"
#include "osl/oslConfig.h"
#include <cassert>
#include <algorithm>
#include <iostream>

#if defined(__linux)
#   include <unistd.h>
#elif defined(__APPLE__)
#   include <mach/mach.h>
#   include <mach/machine.h>
#elif defined(__FreeBSD__)
#   include <sys/types.h>
#   include <sys/sysctl.h>
#   include <unistd.h>
#elif defined(_WIN32)
#   include <windows.h>
#endif

int osl::misc::
ncores() {
  int cpuCount = 1;

#if defined(__linux)
  cpuCount = sysconf(_SC_NPROCESSORS_ONLN);
#elif defined(__APPLE__)
  kern_return_t kr;
  struct host_basic_info hostinfo;
  mach_msg_type_number_t count = HOST_BASIC_INFO_COUNT;
  kr = host_info(mach_host_self(), HOST_BASIC_INFO, (host_info_t)&hostinfo, &count);
  if(kr == KERN_SUCCESS) {
    cpuCount = hostinfo.avail_cpus;
  }
#elif defined( __FreeBSD__)
  cpuCount = sysconf(_SC_NPROCESSORS_ONLN);
#elif defined(_WIN32)
  {
    typedef void (WINAPI *PGNSI)(LPSYSTEM_INFO);
    SYSTEM_INFO si;
    PGNSI pGNSI = NULL;
#   ifndef __MINGW32__
    pGNSI = (PGNSI)GetProcAddress(GetModuleHandle(TEXT("kernel32.dll")),  
                                  "GetNativeSystemInfo"); // The system frees it.
#   endif
    if(NULL != pGNSI){
        pGNSI(&si);
    }else{
        GetSystemInfo(&si);
    }
    cpuCount = si.dwNumberOfProcessors;
  }
#else
  std::cerr << "Unknown #cores. Use the default value: " << cpuCount << "\n";
#endif

  assert(cpuCount > 0);
  if (cpuCount > OslConfig::MaxThreads)
    std::cerr << "cpuCount " << cpuCount << " > " << "MaxThreads " << OslConfig::MaxThreads << "\n";
  return std::min(cpuCount, OslConfig::MaxThreads);
}

/*
 * Alternative is
 *   <sched.h>
 *   CPU_COUNT (cpusetp)
 * It is available for glibc 2.6
 * See http://www.nabble.com/-gomp--Speed-up-and-improve-CPU-number-checking-t3681303.html
 *
 */

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:

