/*****************************************************************************
*
* Copyright (c) 2000 - 2013, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                          avtExtrudedVolWriter.C                           //
// ************************************************************************* //

#include <avtExtrudedVolWriter.h>

#include <snprintf.h>
#include <vector>

#include <vtkCellData.h>
#include <vtkDataArray.h>
#include <vtkDataSet.h>
#include <vtkDataSetWriter.h>
#include <vtkPointData.h>

#include <avtDatabaseMetaData.h>
#include <DBOptionsAttributes.h>

#include <InvalidVariableException.h>


using     std::string;
using     std::vector;


// ****************************************************************************
//  Method: avtExtrudedVolWriter constructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

avtExtrudedVolWriter::avtExtrudedVolWriter(DBOptionsAttributes *atts)
{
    timestep   = atts->GetInt("Time");
    nTimesteps = atts->GetInt("nTimes");
    numChunks  = -1;
}

// ****************************************************************************
//  Method: avtExtrudedVolWriter::OpenFile
//
//  Purpose:
//      Does no actual work.  Just records the stem name for the files.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::OpenFile(const string &stemname, int numblocks)
{
    stem      = stemname;
    numChunks = numblocks;
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::WriteHeaders
//
//  Purpose:
//      Writes out a VisIt file to tie the ExtrudedVol files together.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::WriteHeaders(const avtDatabaseMetaData *md,
                           vector<string> &scalars, vector<string> &vectors,
                           vector<string> &materials)
{
    if (timestep == 0)
    {
        char filename[1024];
        SNPRINTF(filename, 1024, "%s.exvol", stem.c_str());

        ofstream ofile(filename);
        if (ofile.fail())
        {
            EXCEPTION1(VisItException, "Unable to open a file for writing "
                                     "when saving out to ExtrudedVol format.");
        }

        ofile << "STEM: " << stem.c_str() << endl;
        ofile << "NUMCHUNKS: " << numChunks << endl;
        ofile << "NTIMES: " << nTimesteps << endl;
        variables = scalars;
        ofile << "VARIABLES: " << variables.size() << endl;
        for (int i = 0 ; i < variables.size() ; i++)
            if (strncmp(variables[i].c_str(), "AVT_TRACE_HIST_", 
                        strlen("AVT_TRACE_HIST_")) == 0)
                ofile << variables[i].c_str() + strlen("AVT_TRACE_HIST_")<<endl;
            else
                ofile << variables[i] << endl;
    }
    else
    {
        variables = scalars;
    }
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::WriteChunk
//
//  Purpose:
//      This writes out one chunk of an avtDataset.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::WriteChunk(vtkDataSet *ds, int chunk)
{
    int  i, j;

    if (timestep == -1)
    {
        vtkDataSetWriter *wrtr = vtkDataSetWriter::New();
        wrtr->SetInputData(ds);
        char filename[1024];
        SNPRINTF(filename, 1024, "%s.%d.exvol_conn", stem.c_str(), chunk);
        wrtr->SetFileName(filename);
        wrtr->Write();
        wrtr->Delete();
    }
    else
    {
        char filename[1024];
        SNPRINTF(filename, 1024, "%s.%d.%d.exvol_var", stem.c_str(), 
                                                       timestep, chunk);

        ofstream ofile(filename);
        if (ofile.fail())
        {
            EXCEPTION1(VisItException, "Unable to open a file for writing "
                                     "when saving out to ExtrudedVol format.");
        }

        int npts = ds->GetNumberOfPoints();
        ofile << npts << endl;

        for (i = 0 ; i < npts ; i++)
        {
            double pt[3];
            ds->GetPoint(i, pt);
            ofile << pt[0] << " " << pt[1] << " " << pt[2] << endl;
        }

        for (i = 0 ; i < variables.size() ; i++)
        {
            vtkDataArray *arr = 
                            ds->GetPointData()->GetArray(variables[i].c_str());
            if (arr == NULL)
                ds->GetCellData()->GetArray(variables[i].c_str());
            if (arr == NULL)
            {
                EXCEPTION1(InvalidVariableException, variables[i].c_str());
            }
            for (j = 0 ; j < npts ; j++)
                ofile << arr->GetTuple1(j) << endl;
        }
    }
}


// ****************************************************************************
//  Method: avtExtrudedVolWriter::CloseFile
//
//  Purpose:
//      Closes the file.  This does nothing in this case.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri May 18 17:52:04 PST 2007
//
// ****************************************************************************

void
avtExtrudedVolWriter::CloseFile(void)
{
    ;
}


