/*	Animator

PIRL CVS ID: Animator.java,v 1.5 2012/04/16 06:22:58 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Viewers;

import	javax.swing.JLabel;
import	javax.swing.Timer;
import	javax.swing.Icon;
import	javax.swing.ImageIcon;
import	java.awt.Component;
import	java.awt.Graphics;
import	java.awt.event.*;

/**	An <i>Animator</i> is a JLabel that displays an animation of a series
	of Icons.
<p>
	Once started, an Animator will sequentially display the sequence of
	Icons that it has been given. The rate at which Icons are displayed
	and the number of animation cycles is controllable. The animation
	sequence may be stopped and started again from where it left off,
	or restarted from the beginning of a cycle.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.5 
*/
public class Animator
	extends JLabel
	implements ActionListener
{
private static final String
	ID = "PIRL.Viewers.Animator (1.5 2012/04/16 06:22:58)";

/**	Constant to specify an indefinate number of animation cycles;
	i.e&#46; forever until stopped.
*/
public static final int
	INDEFINITELY			=  Integer.MAX_VALUE;

/**	The default number of animation cycles: INDEFINITELY.
*/
public static final int
	CYCLES					= INDEFINITELY;

/**	The default initial delay before animation starts: 0 milliseconds
	(immediately after start).
*/
public static final int
	INITIAL_DELAY			= 0;

/**	The default animation interval: 1000 milliseconds.
*/
public static final int
	INTERVAL				= 1000;

/**	The minimum animation interval: 50 milliseconds.
*/
public static final int
	MINIMUM_INTERVAL		= 50;

private Timer
	Animation_Timer;

private Icon[]
	Icon_Sequence;
private int
	Max_Icon_Width			= 0,
	Max_Icon_Height			= 0;
private boolean
	Adjust_Icon_Location	= false;

private int
	Icon_Index				= 0,
	Cycle					= 0,
	Cycles;


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_TIMER			= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG       		= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Constructs an Icon Animator.
<p>
	@param	icons	An array of Icon objects to be animated.
	@param	cycles	The maximum number of animation cycles. Use the
		<CODE>{@link #INDEFINITELY}</CODE> value to continue the
		animation until stopped.
	@param	initial_delay	The initial delay, after first starting,
		before animation begins. A value <= 0 means no initial delay.
	@param	interval	The interval between the display of each Icon in
		the animation sequence. A {@link #MINIMUM_INTERVAL} is enforced.
*/
public Animator
	(
	Icon[]		icons,
	int			cycles,
	int			initial_delay,
	int			interval
	)
{
super ((icons == null) ? (Icon)null : icons[0]);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">>> Animator:\n"
		+"    " + ((icons == null) ? "null" : String.valueOf (icons.length))
			+ " icons\n"
		+"           cycles - " + cycles + '\n'
		+"    initial_delay - " + initial_delay + '\n'
		+"         interval - " + interval);
if (interval < MINIMUM_INTERVAL)
	interval = MINIMUM_INTERVAL;
if (initial_delay < 0)
	initial_delay = 0;
Animation_Timer = new Timer (interval, this);
Animation_Timer.setInitialDelay (initial_delay);
Animation_Timer.setRepeats (true);
Animation_Timer.setCoalesce (true);
setRepeats (cycles);
Icons (icons);
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		("<<< Animator");
}

/**	Constructs an Animator using all of the default Animator parameters.
<p>
	@param	icons	An array of Icon objects to be animated.
*/
public Animator
	(
	Icon[]	icons
	)
{this (icons, CYCLES, INITIAL_DELAY, INTERVAL);}

/**	Constructs a disabled Animator with default Animator parameters.
*/
public Animator ()
{this (null, CYCLES, INITIAL_DELAY, INTERVAL);}

/*==============================================================================
	Timer
*/
/**	Starts the animation.
<p>
	If the animator was {@link #stop() stopped} in the middle of an
	animation cycle it pickes up from where it left off.
<p>
	Use restart to start the animation from the beginning of a cycle.
*/
public void start ()
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Animator.start");
if (Icon_Sequence != null &&
	Icon_Sequence.length != 0)
	Animation_Timer.start ();
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Animator.start: isRunning " + Animation_Timer.isRunning ());
}

/**	Stops the animation.
*/
public void stop ()
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Animator.stop");
Animation_Timer.stop ();
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Animator.stop");
}

/**	Restarts the Animator at the beginning of a cycle.
*/
public void restart ()
{
stop ();
Cycle = 0;
Icon_Index = 0;
if (Icon_Sequence != null &&
	Icon_Sequence.length != 0)
	setIcon (Icon_Sequence[0]);
start ();
}

/**	Sets the maximum number of animation cycles.
<p>
	@param	cycles	The maximum number of times to cycle through the Icons.
		Use the <CODE>{@link #INDEFINITELY}</CODE> value to continue
		the animation until stopped.
	@return	This Animator.
*/
public Animator setRepeats
	(
	int		cycles
	)
{
int
	repeats = Cycles;
if (cycles < 0)
	cycles = 0;
Cycles = cycles;
return this;
}

/**	Gets the number of animation cycles.
<p>
	@return	The number of times to cycle through the Icon_Sequence
		animation. For indefinate animation this will be the <CODE>{@link
		#INDEFINITELY}</CODE> value.
*/
public int getRepeats ()
{return Cycles;}

/**	Gets the initial delay.
<p>
	@return	The intial delay time in milliseconds.
	@see	#setInitialDelay(int)
*/
public int getInitialDelay ()
{return Animation_Timer.getInitialDelay ();}

/**	Sets the initial delay.
<p>
	The intial delay occurs before the first animation cycle {@link #start()
	starts} or is {@link #restart() restarted}.
<p>
	@param	initial_delay	The intial delay time in milliseconds. A value
		less than or equal to zero means no initial delay.
	@return	This Animator.
*/
public Animator setInitialDelay
	(
	int		initial_delay
	)
{
if (initial_delay < 0)
	initial_delay = 0;
Animation_Timer.setInitialDelay (initial_delay);
return this;
}

/**	Gets the delay between the display of each Icon in the animation
	sequence.
<p>
	@return	The delay between the display of each Icon during an
		animation sequence.
	@see	#setDelay(int)
*/
public int getDelay ()
{return Animation_Timer.getDelay ();}

/**	Set the delay between the display of each Icon in the animation
	sequence.
<p>
	After the animation sequence is {@link #start() started} the next
	Icon is displayed after the interval delay. Because the time can not
	guarantee its accuracy, especially when the display component must be
	rendered during the interval, the interval may be longer than
	expected at times. In general, a longer interval is more likely to
	result in a smoother animation than a short interval.
<p>
	@param	interval	The delay between the display of each Icon during
		an animation sequence. A {@link #MINIMUM_INTERVAL} is enforced.
	@return	This Animator.
*/
public Animator setDelay
	(
	int		interval
	)
{
if (interval < MINIMUM_INTERVAL)
	interval = MINIMUM_INTERVAL;
Animation_Timer.setDelay (interval);
return this;
}

/**	Tests if the animation sequence is running.
<p>
	@return	true if the animation sequence is active; false otherwise.
*/
public boolean isRunning ()
{return Animation_Timer.isRunning ();}

/*==============================================================================
	ActionListener
*/
/**	Performs the animation event at the Timer interval.
<p>
	If the animation is active the next Icon is painted.
*/
public void actionPerformed
	(
	ActionEvent	event
	)
{
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		(">>> Animator.actionPerformed");
if (Icon_Sequence != null &&
	Icon_Sequence.length != 0)
	{
	if (++Icon_Index >= Icon_Sequence.length)
		{
		if (++Cycle > Cycles)
			{
			if ((DEBUG & DEBUG_TIMER) != 0)
				System.out.println
					("    Stopping at cycle " + Cycle + " of " + Cycles);
			Cycle = 0;
			stop ();
			if ((DEBUG & DEBUG_TIMER) != 0)
				System.out.println
					("<<< Animator.actionPerformed");
			return;
			}
		if ((DEBUG & DEBUG_TIMER) != 0)
			System.out.println
				("    Starting cycle " + Cycle);
		Icon_Index = 0;
		}
	if ((DEBUG & DEBUG_TIMER) != 0)
		System.out.println
			("    Displaying Icon " + Icon_Index);
	setIcon (Icon_Sequence[Icon_Index]);
	}
if ((DEBUG & DEBUG_TIMER) != 0)
	System.out.println
		("<<< Animator.actionPerformed");
}

/*==============================================================================
	Accessors
*/
/**	Sets the Icon array to be animated.
<p>
	The animation is {@link #stop() stopped} and the {@link #setRepeats(int)
	cycle} count is
	reset to zero.
<p>
	@param	icons	An array of Icon objects to be animated.
	@return	This Animator.
*/
public Animator Icons
	(
	Icon[]	icons
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Icons");
stop ();
Cycle = 0;
Icon_Sequence = icons;
if (Icon_Sequence != null &&
	Icon_Sequence.length != 0)
	{
	Max_Icon_Width  = Icon_Sequence[0].getIconWidth ();
	Max_Icon_Height = Icon_Sequence[0].getIconHeight ();
	Adjust_Icon_Location = false;
	for (int
			index = 1;
			index < Icon_Sequence.length;
			index++)
		{
		if (Max_Icon_Width < Icon_Sequence[index].getIconWidth ())
			{
			Max_Icon_Width = Icon_Sequence[index].getIconWidth ();
			Adjust_Icon_Location = true;
			}
		if (Max_Icon_Height < Icon_Sequence[index].getIconHeight ())
			{
			Max_Icon_Height = Icon_Sequence[index].getIconHeight ();
			Adjust_Icon_Location = true;
			}
		}
	if ((DEBUG & DEBUG_ACCESSORS) != 0)
		System.out.println
			("    Max Icon size: " + Max_Icon_Width + ',' + Max_Icon_Height);
	}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Icons");
return this;
}

/**	Gets the array of Icons to be animated.
<p>
	@return	The array of animated Icons.
*/
public Icon[] Icons ()
{return Icon_Sequence;}

/**	Gets the current animation step.
<p>
	@return	The current animation step.
*/
public int Step ()
{return Icon_Index;}

/**	Gets the current animation cycle.
<p>
	@return	The current animation cycle count.
*/
public int Cycle ()
{return Cycle;}


}	//	class Animator.
