// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/cr_button/cr_button.m.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.m.js';
import 'chrome://resources/cr_elements/shared_vars_css.m.js';
import './shared_css.js';

import {assert} from 'chrome://resources/js/assert.m.js';
import {focusWithoutInk} from 'chrome://resources/js/cr/ui/focus_without_ink.m.js';
import {I18nMixin} from 'chrome://resources/js/i18n_mixin.js';
import {WebUIListenerMixin} from 'chrome://resources/js/web_ui_listener_mixin.js';
import {afterNextRender, DomRepeatEvent, PolymerElement} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {AutogeneratedThemeColorInfo, AvailableAccount, ManageProfilesBrowserProxy, ManageProfilesBrowserProxyImpl} from '../manage_profiles_browser_proxy.js';
import {hasPreviousRoute, navigateToPreviousRoute} from '../navigation_mixin.js';

import {getTemplate} from './account_selection_lacros.html.js';

export interface AccountSelectionLacrosElement {
  $: {
    backButton: HTMLElement,
    'product-logo': HTMLElement,
  };
}

const AccountSelectionLacrosElementBase =
    WebUIListenerMixin(I18nMixin(PolymerElement));

export class AccountSelectionLacrosElement extends
    AccountSelectionLacrosElementBase {
  static get is() {
    return 'account-selection-lacros';
  }

  static get template() {
    return getTemplate();
  }

  static get properties() {
    return {
      profileThemeInfo: Object,

      availableAccounts_: {
        type: Array,
        value: () => [],
      },

      hasPreviousRoute_: {
        type: Boolean,
        value: () => hasPreviousRoute(),
      },
    };
  }

  private manageProfilesBrowserProxy_: ManageProfilesBrowserProxy =
      ManageProfilesBrowserProxyImpl.getInstance();

  profileThemeInfo: AutogeneratedThemeColorInfo;
  private availableAccounts_: Array<AvailableAccount>;

  override connectedCallback() {
    super.connectedCallback();
    this.addWebUIListener(
        'available-accounts-changed',
        (accounts: Array<AvailableAccount>) =>
            this.handleAvailableAccountsChanged_(accounts));
    this.manageProfilesBrowserProxy_.getAvailableAccounts();
  }

  override ready() {
    super.ready();
    this.addEventListener('view-enter-start', this.onViewEnterStart_);
  }

  private onViewEnterStart_() {
    if (hasPreviousRoute()) {
      afterNextRender(
          this,
          () =>
              focusWithoutInk(this.shadowRoot!.querySelector('#backButton')!));
    }
  }

  private onBackClick_() {
    assert(hasPreviousRoute());
    navigateToPreviousRoute();
  }

  private getBackButtonAriaLabel_(): string {
    return this.i18n(
        'backButtonAriaLabel', this.i18n('accountSelectionLacrosTitle'));
  }

  private onProductLogoTap_() {
    this.$['product-logo'].animate(
        {
          transform: ['none', 'rotate(-10turn)'],
        },
        {
          duration: 500,
          easing: 'cubic-bezier(1, 0, 0, 1)',
        });
  }

  private onAccountClick_(e: DomRepeatEvent<AvailableAccount>) {
    const gaiaId = e.model.item.gaiaId;
    this.manageProfilesBrowserProxy_.loadSignInProfileCreationFlow(
        this.profileThemeInfo.color, gaiaId);
  }

  private onOtherAccountClick_() {
    this.manageProfilesBrowserProxy_.loadSignInProfileCreationFlow(
        this.profileThemeInfo.color, '');
  }

  private handleAvailableAccountsChanged_(availableAccounts:
                                              Array<AvailableAccount>) {
    this.availableAccounts_ = availableAccounts;
  }
}

declare global {
  interface HTMLElementTagNameMap {
    'account-selection-lacros': AccountSelectionLacrosElement;
  }
}

customElements.define(
    AccountSelectionLacrosElement.is, AccountSelectionLacrosElement);
