/**
 * UGENE - Integrated Bioinformatics Tools.
 * Copyright (C) 2008-2011 UniPro <ugene@unipro.ru>
 * http://ugene.unipro.ru
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#ifndef __ASSEMBLY_READS_AREA_H__
#define __ASSEMBLY_READS_AREA_H__

#include <assert.h>

#include <QtGui/QWidget>
#include <QtGui/QScrollBar>
#include <QtGui/QMenu>
#include <QtCore/QSharedPointer>

#include <U2Core/U2Assembly.h>
#include "AssemblyCellRenderer.h"
#include "AssemblyReadsAreaHint.h"
#include "AssemblyModel.h"

namespace U2 {

class AssemblyBrowser;
class AssemblyBrowserUi;
class AssemblyReadsArea;

class AssemblyReadsArea: public QWidget {
    Q_OBJECT
    
    // (hotkey, description) pairs
    struct HotkeyDescription {
        QString key;
        QString desc;
        HotkeyDescription(const QString& k, const QString & d) : key(k), desc(d) {}
    };
    static const QList<HotkeyDescription> HOTKEY_DESCRIPTIONS;
    static QList<HotkeyDescription> initHotkeyDescriptions();
    
public:
    AssemblyReadsArea(AssemblyBrowserUi * ui, QScrollBar * hBar, QScrollBar * vBar);
    
protected:
    void paintEvent(QPaintEvent * e);
    void resizeEvent(QResizeEvent * e);
    void wheelEvent(QWheelEvent * e);
    void mousePressEvent(QMouseEvent * e);
    void mouseReleaseEvent(QMouseEvent * e);
    void mouseMoveEvent(QMouseEvent * e);
    void leaveEvent(QEvent * e);
    void hideEvent(QHideEvent * e);
    bool event(QEvent * e);
    void keyPressEvent(QKeyEvent * e);
    void mouseDoubleClickEvent(QMouseEvent * e);
    bool eventFilter(QObject *obj, QEvent *ev);
    
private:
    void initRedraw();
    void connectSlots();
    void setupHScrollBar();
    void setupVScrollBar();

    void drawAll();
    void drawWelcomeScreen(QPainter & p);
    void drawReads(QPainter & p);
    void drawHint(QPainter & p);
    
    int calcFontPointSize() const;
    
    bool findReadUnderMouse(U2AssemblyRead & read);
    void updateMenuActions();
    void exportReads(const QList<U2AssemblyRead> & reads);
        
signals:
    void si_heightChanged();
    void si_mouseMovedToPos(const QPoint &);
    
public slots:
    void sl_hideHint();  
    void sl_redraw();
    
private slots:
    void sl_coveredRegionClicked(const QString & link);
    void sl_onHScrollMoved(int pos);
    void sl_onVScrollMoved(int pos);
    void sl_zoomOperationPerformed();
    void sl_onCopyReadData();
    void sl_onExportRead();
    void sl_onExportReadsOnScreen();
    
private:
    AssemblyBrowserUi * ui;
    AssemblyBrowser * browser;
    QSharedPointer<AssemblyModel> model;

    bool redraw;
    QPixmap cachedView;
    AssemblyCellRenderer cellRenderer;
    
    QLabel coveredRegionsLabel;
    QScrollBar * hBar;
    QScrollBar * vBar;
    
    // caches reads that are visible on a screen
    class ReadsCache {
    public:
        bool isEmpty() const {
            return data.isEmpty();
        }
        void clear() {
            data.clear();
            visibleBases = U2Region();
            visibleRows = U2Region();
            letterWidth = 0;
            xOffsetInAssembly = 0;
            yOffsetInAssembly = 0;
        }
        QList<U2AssemblyRead> data;
        U2Region visibleBases;
        U2Region visibleRows;
        int letterWidth;
        qint64 xOffsetInAssembly;
        qint64 yOffsetInAssembly;
    };
    ReadsCache cachedReads;
    QPoint curPos;
    
    struct HintData {
        HintData(QWidget * p) : redrawHint(false), hint(p) {}
        bool redrawHint;
        AssemblyReadsAreaHint hint;
        U2DataId curReadId;
    } hintData;
    
    // needed to move by-letter when scribbling
    class ReadsMover {
    public:
        ReadsMover() : cellWidth(0) {};
        ReadsMover(int cellWidth_, QPoint initPos) : lastPos(initPos), cellWidth(cellWidth_ ? cellWidth_ : 1){}
        void handleEvent(QPoint newPos) {
            assert(cellWidth);
            QPoint diff_ = newPos - lastPos;
            lastPos = newPos;
            diff += diff_;
        }
        int getXunits() {
            assert(cellWidth);
            int result = diff.x() / cellWidth;
            diff.setX(diff.x() % cellWidth);
            return result;
        }
        int getYunits() {
            assert(cellWidth);
            int result = diff.y() / cellWidth;
            diff.setY(diff.y() % cellWidth);
            return result;
        }
    private:
        QPoint lastPos;
        QPoint diff;
        int cellWidth;
    } mover;

    bool scribbling;
    
    QMenu * readMenu;
    QAction * copyDataAction;
    QAction * exportReadAction;
    
    int currentHotkeyIndex;
};

} //ns

#endif 
